import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import SolvedAssignmentDetailClient, { ProductData } from "./SolvedAssignmentDetailClient";
import { buildAssignmentDetailMeta } from "@/lib/seo/assignmentMeta";

export const dynamic = "force-dynamic";

async function fetchAssignmentData(courseCode: string, assignmentCode: string) {
    try {
        const [res1, res2, res3, res4] = await Promise.all([
            API_ENDPOINTS.ASSIGNMENTS.DETAIL(courseCode, assignmentCode),
            API_ENDPOINTS.SUBSECTIONS(3, 35),
            API_ENDPOINTS.SUBSECTIONS(3, 36),
            API_ENDPOINTS.SUBSECTIONS(3, 43),
        ]);

        if (!res1.ok) throw new Error(`HTTP ${res1.status}`);

        const [data1, data2, data3, data4] = await Promise.all([
            res1.json(),
            res2.ok ? res2.json() : [],
            res3.ok ? res3.json() : [],
            res4.ok ? res4.json() : [],
        ]);

        return {
            productData: data1.assignment as ProductData,
            firstSection: data2,
            secondSection: data3,
            thirdSection: data4,
        };
    } catch (err) {
        console.error("❌ Failed to load assignment details", err);
        return null;
    }
}

export async function generateMetadata({ params }: { params: Promise<{ slug: string, code: string }> }): Promise<Metadata> {
    const resolvedParams = await params;
    const data = await fetchAssignmentData(resolvedParams.slug, resolvedParams.code);

    if (!data || !data.productData) {
        return { title: "Assignment Detail" };
    }

    const meta = buildAssignmentDetailMeta(data.productData);
    return {
        ...meta,
        alternates: {
            canonical: `/assignments/solved/${resolvedParams.slug}/${resolvedParams.code}`,
        }
    };
}

export default async function AssignmentDetailPage({ params }: { params: Promise<{ slug: string, code: string }> }) {
    const resolvedParams = await params;
    const data = await fetchAssignmentData(resolvedParams.slug, resolvedParams.code);

    if (!data || !data.productData) {
        return <div className="text-danger text-center py-5">Failed to load assignment details.</div>;
    }

    return (
        <SolvedAssignmentDetailClient
            productData={data.productData}
            firstSection={data.firstSection}
            secondSection={data.secondSection}
            thirdSection={data.thirdSection}
            courseCode={resolvedParams.slug}
            assignmentCode={resolvedParams.code}
        />
    );
}